/** @file   glyphset.h
 * @brief   Declaraction of GlypthSet - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_GLYPHSET_H
#define H_ENG2D_GLYPHSET_H

#include <allegro.h>
#include <map>
#include <iostream>
#include "eng2d_dll.h"
#include "glyph.h"


namespace eng2d {


/** @class  GlyphSet
 * @brief   Contains all the graphics that can be printed with this font.
 * @author  Tomi Lamminsaari
 *
 * 
 */
class DLLIMPORT GlyphSet
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     pB                Pointer to bitmap containing the character
   *                              graphics.
   * @param     rIn               Input stream containing
   */
	GlyphSet( BITMAP* pB, std::istream& rIn );


	/** Destructor
   */
	virtual ~GlyphSet();

private:

	/** Copy constructor.
   * @param     rO                Reference to another GlyphSet
   */
  GlyphSet( const GlyphSet& rO );

	/** Assignment operator
   * @param     rO                Reference to another GlyphSet
   * @return    Reference to us.
   */
  GlyphSet& operator = ( const GlyphSet& rO );

public:


  ///
  /// Methods
  /// =======



  ///
  /// Getter methods
  /// ==============

  /** Tells if this glyphset has graphics for given character
   * @param     c                 The character
   * @return    <code>true</code> if we have graphics
   */
  bool hasGlyph( int c ) const;
  
  /** Returns the width of given character
   * @param     c                 The character
   * @return    Width of the character.
   */
  int glyphWidth( int c ) const;
  
  /** Returns the bitmap that contains the graphics of given character.
   * @param     c                 The character
   * @return    Pointer to the graphics or null-pointer if the glyphset does
   *            not have graphics for the given character.
   */
  BITMAP* gfx( int c ) const;
  
  /** Returns the of the glyphset
   * @return    Size of the font in pixels
   */
  int glyphSize() const;
   
  /** Tells if this glyphset is ok. If the loading fails, this returnes
   * true.
   * @return    <code>true</code> if this glyphset is not usable.
   */
  bool bad() const;
  
  /** Returns the ID code of this font.
   * @return    THe id of this font
   */
  std::string id() const;

protected:

  /** Reads the glyph data
   * @param     pB                Pointer to bitmap
   * @param     rIn               The input stream containing the font
   *                              information
   * @return    Nonzero if fails
   */
  int readGlyphData( BITMAP* pB, std::istream& rIn );
  
  /** Reads the glyphset
   * @param     rIn               The input stream to read from
   * @return    Nonzero if fails
   */
  int readGlyphSet( std::istream& rIn );
  
  /** Creates the characters.
   * @param     pB                Pointer to bitmap holding the graphics of
   *                              the characters.
   */
  void createCharacters( BITMAP* pB );
  
  /** Grabs the character from given bitmap
   * @param     pB                pointer to bitmap
   * @param     c                 Index of the character.
   * @return    Pointer to bitmap containing the graphics of the index'th
   *            character.
   */
  BITMAP* grabCharacter( BITMAP* pB, int c );
  
  /** Returns the pointer to glyph holding the data of given character.
   * @param     c                 The character
   * @return    Pointer to Glyph-structure holding the data or nullpointer
   *            if there is no data for given character.
   */
  Glyph* getGlyphData( int c ) const;
  
  
  ///
  /// Members
  /// =======

  /** Width of the grid where the graphics are taken. */
  int m_gridX;
  
  /** Height of the grid where the graphics are taken. */
  int m_gridY;
  
  /** Height of the characters. */
  int m_glyphHeight;
  
  /** ID of this glyphset. */
  std::string m_id;
  
  /** This map holds all the characters that are supported by this GlyphSet.
   * The ASCII - code of the character operates as the key.
   */
  std::map<int, Glyph*> m_characters;
  
  /** Set, if loading fails */
  bool  m_bad;


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
